/*
 * kryptet
 */

import {ValidationUtil} from '@wireapp/commons';
import {app, webContents} from 'electron';
import * as fs from 'fs-extra';
import * as path from 'path';

import {getLogger} from '../logging/getLogger';

const USER_DATA_DIR = app.getPath('userData');
const LOG_DIR = path.join(USER_DATA_DIR, 'logs');

const logger = getLogger(__filename);

const clearStorage = (session: Electron.Session): Promise<void> => {
  return new Promise(resolve =>
    session.clearStorageData({}, () =>
      session.clearCache(() => {
        session.flushStorageData();
        resolve();
      }),
    ),
  );
};

export async function deleteAccount(id: number, accountId: string, partitionId?: string): Promise<void> {
  // Delete session data
  try {
    const webviewWebcontent = webContents.fromId(id);
    if (!webviewWebcontent) {
      throw new Error(`Unable to find webview content id "${id}"`);
    }
    if (!webviewWebcontent.hostWebContents) {
      throw new Error('Only a webview can have its storage wiped');
    }
    logger.log(`Deleting session data for account "${accountId}"...`);
    await clearStorage(webviewWebcontent.session);
    logger.log(`Deleted session data for account "${accountId}".`);
  } catch (error) {
    logger.error(`Failed to delete session data for account "${accountId}", reason: "${error.message}".`);
  }

  // Delete the webview partition
  // Note: The first account always uses the default session,
  // therefore partitionId is optional
  // ToDo: Move the first account to a partition
  if (partitionId) {
    try {
      if (!ValidationUtil.isUUIDv4(partitionId)) {
        throw new Error('Partition is not an UUID');
      }
      const partitionDir = path.join(USER_DATA_DIR, 'Partitions', partitionId);
      await fs.remove(partitionDir);
      logger.log(`Deleted partition "${partitionId}" for account "${accountId}".`);
    } catch (error) {
      logger.log(`Unable to delete partition "${partitionId}" for account "${accountId}", reason: "${error.message}".`);
    }
  }

  // Delete logs for this account
  try {
    if (!ValidationUtil.isUUIDv4(accountId)) {
      throw new Error('Account is not an UUID');
    }
    const sessionFolder = path.join(LOG_DIR, accountId);
    await fs.remove(sessionFolder);
    logger.log(`Deleted logs folder for account "${accountId}".`);
  } catch (error) {
    logger.error(`Failed to delete logs folder for account "${accountId}", reason: "${error.message}".`);
  }
}
