/*
 * kryptet
 */

import * as assert from 'assert';
import {BrowserWindow, Tray, app} from 'electron';
import * as path from 'path';
import * as sinon from 'sinon';

import {TrayHandler} from './TrayHandler';

const TrayMock = new Tray(path.join(__dirname, '../../test/fixtures/tray.png'));

describe('initTray', () => {
  it('creates native images for all tray icons and sets a default tray icon', () => {
    const tray = new TrayHandler();
    tray.initTray(TrayMock);
    assert.strictEqual(Object.keys(tray.icons!).length, 3);
    assert.strictEqual(tray.icons!.badge.constructor.name, 'NativeImage');
    assert.strictEqual(tray.icons!.tray.constructor.name, 'NativeImage');
    assert.strictEqual(tray.icons!.trayWithBadge.constructor.name, 'NativeImage');
    sinon.assert.match(tray.trayIcon, sinon.match.defined);
  });
});

describe('showUnreadCount', () => {
  describe('without tray icon initialization', () => {
    it('updates the badge counter and stops flashing the app frame when app is in focus while receiving new messages', done => {
      const tray = new TrayHandler();
      const appWindow = new BrowserWindow();

      const badgeCountSpy = sinon.spy(app, 'setBadgeCount');
      const flashFrameSpy = sinon.spy(appWindow, 'flashFrame');

      appWindow.loadURL('about:blank');
      appWindow.webContents.on('dom-ready', () => {
        assert.strictEqual(appWindow.isFocused(), true);
        assert.ok(flashFrameSpy.notCalled);
        tray.showUnreadCount(appWindow, 1);

        assert.ok(badgeCountSpy.firstCall.calledWith(1));
        assert.ok(flashFrameSpy.firstCall.calledWith(false));
        assert.strictEqual(tray.lastUnreadCount, 1);

        flashFrameSpy.restore();
        badgeCountSpy.restore();
        done();
      });
    });
  });

  describe('with tray icon initialization', () => {
    it('updates the badge counter and stops flashing the app frame when app is in focus while receiving new messages', done => {
      const tray = new TrayHandler();
      tray.initTray(TrayMock);
      const appWindow = new BrowserWindow();

      const flashFrameSpy = sinon.spy(appWindow, 'flashFrame');

      appWindow.loadFile(path.join(__dirname, '../fixtures/badge.html'));
      appWindow.webContents.on('dom-ready', () => {
        assert.strictEqual(appWindow.isFocused(), true);
        assert.ok(flashFrameSpy.notCalled);
        tray.showUnreadCount(appWindow, 10);
        assert.ok(flashFrameSpy.firstCall.calledWith(false));
        assert.strictEqual(tray.lastUnreadCount, 10);
        flashFrameSpy.restore();
        done();
      });
    });

    it('flashes the app frame when app is not in focus and you receive new messages', done => {
      const tray = new TrayHandler();
      tray.initTray(TrayMock);

      const appWindow = new BrowserWindow({
        show: false,
        useContentSize: true,
      });

      const flashFrameSpy = sinon.spy(appWindow, 'flashFrame');

      appWindow.loadURL('about:blank');
      appWindow.webContents.on('dom-ready', () => {
        assert.strictEqual(appWindow.isFocused(), false);
        tray.showUnreadCount(appWindow, 2);
        assert.ok(flashFrameSpy.firstCall.calledWith(true));
        flashFrameSpy.restore();
        done();
      });
      appWindow.showInactive();
    });

    it('does change the flash state if the window has already been flashed', done => {
      const tray = new TrayHandler();
      tray.initTray(TrayMock);
      tray.lastUnreadCount = 5;

      const appWindow = new BrowserWindow({
        show: false,
        useContentSize: true,
      });

      const flashFrameSpy = sinon.spy(appWindow, 'flashFrame');

      appWindow.loadURL('about:blank');
      appWindow.webContents.on('dom-ready', () => {
        assert.strictEqual(appWindow.isFocused(), false);
        tray.showUnreadCount(appWindow, 2);
        assert.ok(flashFrameSpy.notCalled);
        flashFrameSpy.restore();
        done();
      });
      appWindow.showInactive();
    });
  });
});
